<?php
/**
 * File: Extension_ImageService_Page_View.php
 *
 * View for the Image Service extension settings, tools, and statistics page.
 *
 * @since 2.2.0
 *
 * @package W3TC
 *
 * @uses Config      $c      Configuration object.
 * @uses array       $counts Image Service media counts.
 * @uses array|false $usage  API usage statistics.
 */

namespace W3TC;

if ( ! defined( 'W3TC' ) ) {
	die();
}

$c      = Dispatcher::config();
$is_pro = Util_Environment::is_w3tc_pro( $c );
$usage  = Extension_ImageService_Plugin::get_api()->get_usage();
?>
<div class="wrap" id="w3tc">
<?php Util_Ui::print_breadcrumb(); ?>
<p>
	Total Cache Image Converter is currently
<?php
if ( $c->is_extension_active( 'imageservice' ) ) {
	?>
	<span class="w3tc-enabled">enabled</span>
	<?php
} else {
	?>
	<span class="w3tc-disabled">disabled</span>
	<?php
}
?>
	.
</p>

<form id="w3tc-imageservice-settings" action="upload.php?page=w3tc_extension_page_imageservice" method="post">
	<?php Util_UI::print_control_bar( 'extension_imageservice_form_control' ); ?>
<div class="metabox-holder">

	<?php Util_Ui::postbox_header( esc_html__( 'Configuration', 'w3-total-cache' ), '', 'configuration' ); ?>

	<table class="form-table" id="w3tc-imageservice-config">
<?php
Util_Ui::config_item(
	array(
		'key'               => array(
			'imageservice',
			'compression',
		),
		'label'             => esc_html__( 'Compression type:', 'w3-total-cache' ),
		'control'           => 'radiogroup',
		'radiogroup_values' => array(
			'lossy'    => 'Lossy',
			'lossless' => 'Lossless',
		),
		'description'       => esc_html__( 'Image compression type.', 'w3-total-cache' ),
		'disabled'          => false,
	)
);

Util_Ui::config_item(
	array(
		'key'               => array(
			'imageservice',
			'auto',
		),
		'label'             => esc_html__( 'Auto-convert:', 'w3-total-cache' ),
		'control'           => 'radiogroup',
		'radiogroup_values' => array(
			'enabled'  => 'Enabled',
			'disabled' => 'Disabled',
		),
		'description'       => esc_html__( 'Auto-convert images on upload.', 'w3-total-cache' ),
		'disabled'          => false,
	)
);

Util_Ui::config_item(
	array(
		'key'              => array(
			'imageservice',
			'visibility',
		),
		'label'            => esc_html__( 'Visibility:', 'w3-total-cache' ),
		'control'          => 'selectbox',
		'selectbox_values' => array(
			'never'     => array( 'label' => __( 'Never', 'w3-total-cache' ) ),
			'extension' => array( 'label' => __( 'If extension is active', 'w3-total-cache' ) ),
			'always'    => array( 'label' => __( 'Always', 'w3-total-cache' ) ),
		),
		'description'      => esc_html__( 'Show converted image attachments in the Media Library.', 'w3-total-cache' ),
		'disabled'         => false,
	)
);

$settings = $c->get_array( 'imageservice' );
// Default to true for webp if not set (backward compatibility).
$webp_enabled = isset( $settings['webp'] ) ? (bool) $settings['webp'] : true;
// Default to true for avif if not set.  We need to make sure this is true for the Pro version.
$avif_enabled = isset( $settings['avif'] ) ? (bool) $settings['avif'] : true;

Util_Ui::config_item(
	array(
		'key'            => array(
			'imageservice',
			'webp',
		),
		'label'          => esc_html__( 'Output formats:', 'w3-total-cache' ),
		'control'        => 'checkbox',
		'checkbox_label' => esc_html__( 'WebP', 'w3-total-cache' ),
		'value'          => $webp_enabled,
		'description'    => esc_html__( 'Convert images to WebP format.', 'w3-total-cache' ),
		'disabled'       => false,
	)
);

Util_Ui::config_item_pro(
	array(
		'key'            => array(
			'imageservice',
			'avif',
		),
		'label'          => ' ',
		'control'        => 'checkbox',
		'checkbox_label' => esc_html__( 'AVIF', 'w3-total-cache' ),
		'value'          => $avif_enabled,
		'disabled'       => ! $is_pro,
		'excerpt'        => esc_html__( 'Convert images to AVIF format for even better compression and performance.', 'w3-total-cache' ),
		'description'    => array(
			esc_html__( 'AVIF (AV1 Image File Format) is a modern image format that provides superior compression compared to WebP and traditional formats like JPEG and PNG. With AVIF conversion, you can achieve significantly smaller file sizes while maintaining high image quality, resulting in faster page load times and improved user experience.', 'w3-total-cache' ),
		),
		'wrap_separate'  => true,
	)
);
?>
	</table>

<?php
Util_Ui::postbox_footer();

Util_Ui::postbox_header( esc_html__( 'Tools', 'w3-total-cache' ), '', 'tools' );

if ( ! $is_pro ) {
	?>
	<div class="w3tc-gopro-manual-wrap">
		<?php
		Util_Ui::pro_wrap_maybe_start();
		echo wp_kses(
			sprintf(
				// translators: 1 opening HTML p tag, 2 free user hourly limit, 3 free user monthly limit, 4 two HTML br tags, 5 pro user hourly limit, 6 closing HTML p tag.
				__(
					'%1$sFree license users will have a conversion limit of %2$d per hour and %3$d per month.%4$sPro license users will have conversion queue priority as well as a conversion limit of %5$d per hour and unlimited per month.%6$s',
					'w3-total-cache'
				),
				'<p>',
				$usage['limit_hourly_unlicensed'],
				$usage['limit_monthly_unlicensed'],
				'<br/><br/>',
				$usage['limit_hourly_licensed'],
				'</p>'
			),
			array(
				'p'  => array(),
				'br' => array(),
			)
		);
		Util_Ui::print_score_block(
			__( 'Potential Google PageSpeed Gain', 'w3-total-cache' ),
			'+9',
			__( 'Points', 'w3-total-cache' ),
			__( 'In one recent test, converting images to modern formats like WebP or AVIF added over 9 points to the Google PageSpeed score!', 'w3-total-cache' ),
			'https://www.boldgrid.com/support/w3-total-cache/pagespeed-tests/webp/?utm_source=w3tc&utm_medium=webp&utm_campaign=proof'
		);
		Util_Ui::pro_wrap_maybe_end( 'imageservice_settings', false );
		?>
	</div>
	<?php
}
?>

	<table class="form-table" id="w3tc-imageservice-tools">
<?php
Util_Ui::config_item(
	array(
		'key'         => null,
		'label'       => esc_html__( 'Convert all images:', 'w3-total-cache' ),
		'label_class' => 'w3tc-imageservice-all',
		'control'     => 'button',
		'none_label'  => 'Convert All',
		'description' => esc_html__( 'Convert all images in the media library.', 'w3-total-cache' ),
	)
);

Util_Ui::config_item(
	array(
		'key'         => null,
		'label'       => esc_html__( 'Revert all images:', 'w3-total-cache' ),
		'label_class' => 'w3tc-imageservice-revertall',
		'control'     => 'button',
		'none_label'  => 'Revert All',
		'description' => esc_html__( 'Revert all converted images in the media library.', 'w3-total-cache' ),
	)
);
?>
	</table>

<?php

Util_Ui::postbox_footer();

Util_Ui::postbox_header(
	esc_html__( 'Statistics', 'w3-total-cache' ),
	'',
	'statistics'
);

?>

	<table class="form-table" id="w3tc-imageservice-stats">
		<tr>
			<th><?php esc_html_e( 'Counts and file sizes by status:', 'w3-total-cache' ); ?></th>
			<td>
				<table id="w3tc-imageservice-counts">
					<tr>
						<td><?php esc_html_e( 'Total:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-total"><?php echo esc_html( $counts['total'] ); ?></td>
						<td id="w3tc-imageservice-totalbytes"><?php echo esc_html( size_format( $counts['totalbytes'], 2 ) ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Converted:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-converted"><?php echo esc_html( $counts['converted'] ); ?></td>
						<td id="w3tc-imageservice-convertedbytes"><?php echo esc_html( size_format( $counts['convertedbytes'], 2 ) ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Sending:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-sending"><?php echo esc_html( $counts['sending'] ); ?></td>
						<td id="w3tc-imageservice-sendingbytes"><?php echo esc_html( size_format( $counts['sendingbytes'], 2 ) ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Processing:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-processing"><?php echo esc_html( $counts['processing'] ); ?></td>
						<td id="w3tc-imageservice-processingbytes"><?php echo esc_html( size_format( $counts['processingbytes'], 2 ) ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Not converted:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-notconverted"><?php echo esc_html( $counts['notconverted'] ); ?></td>
						<td id="w3tc-imageservice-notconvertedbytes"><?php echo esc_html( size_format( $counts['notconvertedbytes'], 2 ) ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Unconverted:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-unconverted"><?php echo esc_html( $counts['unconverted'] ); ?></td>
						<td id="w3tc-imageservice-unconvertedbytes"><?php echo esc_html( size_format( $counts['unconvertedbytes'], 2 ) ); ?></td>
					</tr>
					<tr><td height="10"></td></tr>
					<tr>
						<td colspan="3"><input id="w3tc-imageservice-refresh-counts" class="button" type="button" value="<?php esc_attr_e( 'Refresh', 'w3-total-cache' ); ?>" /></td>
					</tr>
				</table>
			</td>
		</tr>
		<tr>
			<th><?php esc_html_e( 'Image Converter API usage:', 'w3-total-cache' ); ?></th>
			<td>
				<table id="w3tc-imageservice-usage">
					<tr>
						<td><?php esc_html_e( 'Hourly requests:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-usage-hourly"><?php echo esc_html( $usage['usage_hourly'] ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Hourly limit:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-limit-hourly"><?php echo esc_html( $usage['limit_hourly'] ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Monthly requests:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-usage-monthly"><?php echo esc_html( $usage['usage_monthly'] ); ?></td>
					</tr>
					<tr>
						<td><?php esc_html_e( 'Monthly limit:', 'w3-total-cache' ); ?></td>
						<td id="w3tc-imageservice-limit-monthly"><?php echo esc_html( $usage['limit_monthly'] ); ?></td>
					</tr>
					<tr><td height="10"></td></tr>
					<tr>
						<td colspan="3"><input id="w3tc-imageservice-refresh-usage" class="button" type="button" value="<?php esc_attr_e( 'Refresh', 'w3-total-cache' ); ?>" /></td>
					</tr>
				</table>
			</td>
		</tr>
	</table>

<?php Util_Ui::postbox_footer(); ?>

</div>
</form>
